#region Includes
using GroundhogLearning.GameKit.Utils;
using UnityEngine;
using UnityEngine.Events;
#endregion

namespace GroundhogLearning.GameKit
{
    [AddComponentMenu(EditorMenuConfig.CATEGORY_TIME + "Countdown Timer")]
    public class CountdownTimer : MonoBehaviour
    {
        #region Variables

        [Header("Configuration")]
        [SerializeField, Tooltip("If true, the timer starts automatically when the scene starts.")]
        private bool _autoStart = true;

        [SerializeField, Tooltip("The total duration of the countdown in seconds.")]
        private float _duration = 60f;

        [Header("Events")]
        [SerializeField, Tooltip("Invoked every frame with the remaining time value (in seconds).")]
        private UnityEvent<float> OnTimeValueChanged;

        [SerializeField, Tooltip("Invoked every frame with the formatted remaining time value (MM:SS).")]
        private UnityEvent<string> OnFormattedTimeValueChanged;

        [SerializeField, Tooltip("Invoked when the timer reaches zero.")]
        private UnityEvent OnTimerCompleted;

        private bool _isRunning;
        private float _remainingTime;
        private bool _hasCompleted;

        #endregion


        private void Start()
        {
            if (_autoStart)
            {
                StartTimer();
            }
        }
        private void Update()
        {
            if (!_isRunning || _hasCompleted) return;

            _remainingTime -= Time.deltaTime;

            if (_remainingTime < 0f)
            {
                Debug.Log($"CountdownTimer | {gameObject.name} is complete.");

                _remainingTime = 0f;
                _isRunning = false;
                _hasCompleted = true;
                OnTimerCompleted?.Invoke();
            }


            OnTimeValueChanged?.Invoke(_remainingTime);
            OnFormattedTimeValueChanged?.Invoke(FormatTime(_remainingTime));
        }

        private static string FormatTime(float time)
        {
            var minutes = Mathf.FloorToInt(time / 60f);
            var seconds = Mathf.FloorToInt(time % 60f);
            return $"{minutes:00}:{seconds:00}";
        }

        public void StartTimer()
        {
            _isRunning = true;
            _hasCompleted = false;
            _remainingTime = _duration;
        }
        public void PauseTimer()
        {
            _isRunning = false;
        }
        public void ResetTimer()
        {
            _remainingTime = _duration;
            _hasCompleted = false;
        }
        public void SetDuration(float newDuration)
        {
            _duration = newDuration;
            _remainingTime = newDuration;
        }
    }
}
